<?php
namespace QodeCore\CPT\Shortcodes\Button;

use QodeCore\Lib;

class Button implements Lib\ShortcodeInterface
{
    private $base;

    public function __construct() {
        $this->base = 'qodef_button';

        add_action('vc_before_init', array($this, 'vcMap'));
    }

    public function getBase() {
        return $this->base;
    }

    public function vcMap() {
        if (function_exists('vc_map')) {
            vc_map(
                array(
                    'name'                      => esc_html__('Select Button', 'select-core'),
                    'base'                      => $this->base,
                    'category'                  => esc_html__('by Select', 'select-core'),
                    'icon'                      => 'icon-wpb-button extended-custom-icon',
                    'allowed_container_element' => 'vc_row',
                    'params'                    => array_merge(
                        array(
                            array(
                                'type'        => 'textfield',
                                'param_name'  => 'custom_class',
                                'heading'     => esc_html__('Custom CSS Class', 'select-core'),
                                'description' => esc_html__('Style particular content element differently - add a class name and refer to it in custom CSS', 'select-core')
                            ),
                            array(
                                'type'        => 'dropdown',
                                'param_name'  => 'type',
                                'heading'     => esc_html__('Type', 'select-core'),
                                'value'       => array(
                                    esc_html__('Animated Solid', 'select-core')  => 'animated_solid',
                                    esc_html__('Solid', 'select-core')           => 'solid',
                                    esc_html__('Outline', 'select-core')         => 'outline',
                                    esc_html__('Simple', 'select-core')          => 'simple',
                                    esc_html__('Box', 'select-core')             => 'box',
                                    esc_html__('Icon Box', 'select-core')        => 'icon_box',
                                ),
                                'admin_label' => true
                            ),
                            array(
                                'type'       => 'dropdown',
                                'param_name' => 'size',
                                'heading'    => esc_html__('Size', 'select-core'),
                                'value'      => array(
                                    esc_html__('Default', 'select-core') => '',
                                    esc_html__('Small', 'select-core')   => 'small',
                                    esc_html__('Medium', 'select-core')  => 'medium',
                                    esc_html__('Large', 'select-core')   => 'large',
                                    esc_html__('Huge', 'select-core')    => 'huge'
                                ),
                                'dependency' => array('element' => 'type', 'value' => array('solid', 'animated_solid', 'outline'))
                            ),
                            array(
                                'type'        => 'textfield',
                                'param_name'  => 'text',
                                'heading'     => esc_html__('Text', 'select-core'),
                                'value'       => esc_html__('Button Text', 'select-core'),
                                'save_always' => true,
                                'admin_label' => true,
                                'dependency'  => array('element' => 'type', 'value' => array('solid', 'animated_solid', 'outline', 'simple', 'box'))
                            ),
                            array(
                                'type'       => 'textfield',
                                'param_name' => 'link',
                                'heading'    => esc_html__('Link', 'select-core')
                            ),
                            array(
                                'type'        => 'dropdown',
                                'param_name'  => 'target',
                                'heading'     => esc_html__('Link Target', 'select-core'),
                                'value'       => array_flip(synergia_select_get_link_target_array()),
                                'save_always' => true
                            )
                        ),
                        synergia_select_icon_collections()->getVCParamsArray(array(), '', true),
                        array(
                            array(
                                'type'       => 'colorpicker',
                                'param_name' => 'color',
                                'heading'    => esc_html__('Color', 'select-core'),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'       => 'colorpicker',
                                'param_name' => 'hover_color',
                                'heading'    => esc_html__('Hover Color', 'select-core'),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'       => 'colorpicker',
                                'param_name' => 'background_color',
                                'heading'    => esc_html__('Background Color', 'select-core'),
                                'dependency' => array('element' => 'type', 'value' => array('solid', 'animated_solid')),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'       => 'colorpicker',
                                'param_name' => 'hover_background_color',
                                'heading'    => esc_html__('Hover Background Color', 'select-core'),
                                'dependency' => array('element' => 'type', 'value' => array('solid', 'animated_solid', 'outline')),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'       => 'colorpicker',
                                'param_name' => 'border_color',
                                'heading'    => esc_html__('Border Color', 'select-core'),
                                'dependency' => array('element' => 'type', 'value' => array('solid', 'outline')),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'       => 'colorpicker',
                                'param_name' => 'hover_border_color',
                                'heading'    => esc_html__('Hover Border Color', 'select-core'),
                                'dependency' => array('element' => 'type', 'value' => array('solid', 'outline')),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'       => 'textfield',
                                'param_name' => 'font_size',
                                'heading'    => esc_html__('Font Size (px)', 'select-core'),
                                'group'      => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'        => 'dropdown',
                                'param_name'  => 'font_weight',
                                'heading'     => esc_html__('Font Weight', 'select-core'),
                                'value'       => array_flip(synergia_select_get_font_weight_array(true)),
                                'save_always' => true,
                                'group'       => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'        => 'dropdown',
                                'param_name'  => 'text_transform',
                                'heading'     => esc_html__('Text Transform', 'select-core'),
                                'value'       => array_flip(synergia_select_get_text_transform_array(true)),
                                'save_always' => true
                            ),
                            array(
                                'type'        => 'textfield',
                                'param_name'  => 'margin',
                                'heading'     => esc_html__('Margin', 'select-core'),
                                'description' => esc_html__('Insert margin in format: top right bottom left (e.g. 10px 5px 10px 5px)', 'select-core'),
                                'group'       => esc_html__('Design Options', 'select-core')
                            ),
                            array(
                                'type'        => 'textfield',
                                'param_name'  => 'padding',
                                'heading'     => esc_html__('Button Padding', 'select-core'),
                                'description' => esc_html__('Insert padding in format: top right bottom left (e.g. 10px 5px 10px 5px)', 'select-core'),
                                'dependency'  => array('element' => 'type', 'value' => array('solid', 'animated_solid', 'outline')),
                                'group'       => esc_html__('Design Options', 'select-core')
                            )
                        )
                    )
                )
            );
        }
    }

    public function render($atts, $content = null) {
        $default_atts = array(
            'size'                   => '',
            'type'                   => 'animated_solid',
            'text'                   => '',
            'link'                   => '',
            'target'                 => '_self',
            'color'                  => '',
            'hover_color'            => '',
            'background_color'       => '',
            'hover_background_color' => '',
            'border_color'           => '',
            'hover_border_color'     => '',
            'font_size'              => '',
            'font_weight'            => '',
            'text_transform'         => '',
            'margin'                 => '',
            'padding'                => '',
            'custom_class'           => '',
            'html_type'              => 'anchor',
            'input_name'             => '',
            'custom_attrs'           => array()
        );
        $default_atts = array_merge($default_atts, synergia_select_icon_collections()->getShortcodeParams());
        $params = shortcode_atts($default_atts, $atts);

        if ($params['html_type'] !== 'input') {
            $iconPackName = synergia_select_icon_collections()->getIconCollectionParamNameByKey($params['icon_pack']);
            $params['icon'] = $iconPackName ? $params[$iconPackName] : '';
        }

        if ($params['type'] == 'icon_box' && (empty($params['icon_pack']) || empty($params['icon']))) {
            $params['icon_pack'] = 'dripicons';
            $params['icon'] = 'dripicons-arrow-thin-right';
            $params['dripicon'] = 'dripicons-arrow-thin-right';
        }

        $params['size'] = !empty($params['size']) ? $params['size'] : 'medium';
        $params['type'] = !empty($params['type']) ? $params['type'] : 'animated_solid';

        $params['link'] = !empty($params['link']) ? $params['link'] : '#';
        $params['target'] = !empty($params['target']) ? $params['target'] : $default_atts['target'];

        $params['button_classes'] = $this->getButtonClasses($params);
        $params['button_text_classes'] = $this->getButtonTextClasses($params);
        $params['button_custom_attrs'] = !empty($params['custom_attrs']) ? $params['custom_attrs'] : array();
        $params['button_styles'] = $this->getButtonStyles($params);
        $params['button_overlay_styles'] = $this->getButtonOverlayStyles($params);
        $params['button_data'] = $this->getButtonDataAttr($params);

        return qodef_core_get_shortcode_module_template_part('templates/' . $params['html_type'], 'button', '', $params);
    }

    private function getButtonStyles($params) {
        $styles = array();

        if (!empty($params['color'])) {
            $styles[] = 'color: ' . $params['color'];
        }

        if (!empty($params['background_color']) && $params['type'] !== 'outline') {
            $styles[] = 'background-color: ' . $params['background_color'];
        }

        if (!empty($params['border_color']) && $params['type'] !== 'animated_solid') {
            $styles[] = 'border-color: ' . $params['border_color'];
        }

        if (!empty($params['font_size'])) {
            $styles[] = 'font-size: ' . synergia_select_filter_px($params['font_size']) . 'px';
        }

        if (!empty($params['font_weight']) && $params['font_weight'] !== '') {
            $styles[] = 'font-weight: ' . $params['font_weight'];
        }

        if (!empty($params['text_transform'])) {
            $styles[] = 'text-transform: ' . $params['text_transform'];
        }

        if ($params['margin'] !== '') {
            $styles[] = 'margin: ' . $params['margin'];
        }

        if ($params['padding'] !== '') {
            $styles[] = 'padding: ' . $params['padding'];
        }

        return $styles;
    }

    private function getButtonOverlayStyles($params) {
        if (!empty($params['background_color']) && $params['type'] !== 'outline') {
            $styles[] = 'background-color: ' . $params['background_color'];
        }
    }

    private function getButtonDataAttr($params) {
        $data = array();

        if (!empty($params['hover_color'])) {
            $data['data-hover-color'] = $params['hover_color'];
        }

        if (!empty($params['hover_background_color'])) {
            $data['data-hover-bg-color'] = $params['hover_background_color'];
        }

        if (!empty($params['hover_border_color'])) {
            $data['data-hover-border-color'] = $params['hover_border_color'];
        }

        return $data;
    }

    private function getButtonTextClasses($params) {
        $classes = array('qodef-btn-text');

        if (in_array($params['type'], array('simple', 'icon_box', 'box'))) {
            $classes[] = 'qodef-gradient-vertical-text';
        }

        return $classes;
    }

    private function getButtonClasses($params) {
        $buttonClasses = array(
            'qodef-btn',
            'qodef-btn-' . $params['size'],
            'qodef-btn-' . $params['type']
        );

        if (!empty($params['hover_background_color'])) {
            $buttonClasses[] = 'qodef-btn-custom-hover-bg';
        }

        if (!empty($params['hover_border_color'])) {
            $buttonClasses[] = 'qodef-btn-custom-border-hover';
        }

        if (!empty($params['hover_color'])) {
            $buttonClasses[] = 'qodef-btn-custom-hover-color';
        }

        if (!empty($params['icon'])) {
            $buttonClasses[] = 'qodef-btn-icon';
        }

        if (!empty($params['custom_class'])) {
            $buttonClasses[] = esc_attr($params['custom_class']);
        }

        return $buttonClasses;
    }
}